﻿using System;
using System.Collections;
using System.Text.RegularExpressions;

namespace Jeneva.Validation.Impl
{
    /// <summary>
    /// Represents validation checker helper class
    /// </summary>
    public class Checker : IChecker
    {
        /// <summary>
        /// Checks if field value is assigned
        /// </summary>
        /// <param name="fieldName">fielв name</param>
        /// <param name="target">target DTO object</param>
        /// <returns>true if assigned</returns>
        public bool IsAssigned(string fieldName, Dtobase target)
        {
            return target.IsFieldAssigned(fieldName);
        }

        /// <summary>
        /// Checks if field value was correctly parsed
        /// </summary>
        /// <param name="fieldName">fielв name</param>
        /// <param name="target">target DTO object</param>
        /// <returns>true if parsed correctly</returns>
        public bool IsValidFormat(string fieldName, Dtobase target)
        {
            return !target.IsFieldWrong(fieldName);
        }

        /// <summary>
        /// Checks if field value is NULL
        /// </summary>
        /// <param name="value">field value</param>
        /// <returns>true if NULL</returns>
        public bool IsNull(object value)
        {
            return null == value;
        }

        /// <summary>
        /// Checks if field value equals to m
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="value">field value</param>
        /// <returns>true if equals</returns>
        public bool IsEqualTo<T>(T m, T value) where T : IEquatable<T>
        {
            return m.Equals(value);
        }

        /// <summary>
        /// Checks if field value is equal to one of values
        /// </summary>
        /// <param name="values">values</param>
        /// <param name="value">field value</param>
        /// <returns>true if equals</returns>
        public bool IsEqualToOneOf<T>(T[] values, T value) where T : IEquatable<T>
        {
            bool success = false;
            for (int i = 0; i < values.Length; i++)
            {
                if (value.Equals(values[i]))
                {
                    success = true;
                    break;
                }
            }

            return success;
        }

        /// <summary>
        /// Checks if field value is empty string
        /// </summary>
        /// <param name="value">value</param>
        /// <returns>true if empty</returns>
        public bool IsEmptyString(string value)
        {
            return string.Equals(string.Empty, value);
        }

        /// <summary>
        /// Checks if field value length is between min and max values
        /// </summary>
        /// <param name="min">min</param>
        /// <param name="max">max</param>
        /// <param name="value">value</param>
        /// <returns>true if is between</returns>
        public bool IsLengthBetween(int min, int max, string value)
        {
            return value.Length >= min && value.Length <= max;
        }

        /// <summary>
        /// Checks if field value is less than or equal to 'm'
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="value">value</param>
        /// <returns>true if less or equal</returns>
        public bool IsLessOrEqualTo<T>(T m, T value) where T : struct, IComparable<T>
        {
            return value.CompareTo(m) <= 0;
        }

        /// <summary>
        /// Checks if field value is less than 'm'. IComparable.
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="value">value</param>
        /// <returns>true if less</returns>
        public bool IsLessThan<T>(T m, T value) where T : struct, IComparable<T>
        {
            return value.CompareTo(m) < 0;
        }

        /// <summary>
        /// Checks if field value is greater than or equal to 'm'. IComparable.
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="value">value</param>
        /// <returns>true if greater or equal</returns>
        public bool IsGreaterOrEqualTo<T>(T m, T value) where T : struct, IComparable<T>
        {
            return value.CompareTo(m) >= 0;
        }

        /// <summary>
        /// Checks if field value is greater than 'm'. IComparable.
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="value">value</param>
        /// <returns>true if greater</returns>
        public bool IsGreaterThan<T>(T m, T value) where T : struct, IComparable<T>
        {
            return value.CompareTo(m) > 0;
        }

        /// <summary>
        /// Checks if field value collection size is between min and max values inclusively (field must implement ICollection)
        /// </summary>
        /// <param name="min">min</param>
        /// <param name="max">max</param>
        /// <param name="value">field value</param>
        /// <returns>true if is between</returns>
        public bool IsCountBetween(int min, int max, ICollection value)
        {
            return value.Count >= min && value.Count <= max;
        }

        /// <summary>
        /// Checks if field value qualifies to regular expression
        /// </summary>
        /// <param name="expr">regular expression</param>
        /// <param name="value">field value</param>
        /// <returns>true if qualifies</returns>
        public bool IsRegex(string expr, string value)
        {
            Match match = Regex.Match(value, expr, RegexOptions.IgnoreCase);
            return match.Success;
        }
    }
}